// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_GAL_RECEIVER_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_GAL_RECEIVER_H

#include "common.h"
#include "ChannelManager.h"
#include "Controller.h"
#include "MessageRouter.h"

/**
 * This class is the fundamental building block of the Google Automotive Link protocol.
 * <br>
 * Initializing the receiver library consists of the following steps.
 * <pre>
 *      GalReceiver galReceiver;
 *      // Create an instance of type IControllerCallbacks.
 *      galReceiver.init(controllerCallbacks);
 *      galReceiver.setIdentityInfo(...);
 *      galReceiver.setClientCreds(...);
 *      galReceiver.setDriverPosition(...);
 *      // Repeat this for every endpoint you want.
 *      // Create and start service. Look at the documentation for the following for details.
 *      // AudioSink
 *      // AudioSource
 *      // BluetoothEndpoint
 *      // InputSource
 *      // InstrumentClusterEndpoint
 *      // SensorSource
 *      // RadioEndpoint
 *      // VendorExtension
 *      // VideoSink
 *      galReceiver.registerService()
 *      galReceiver.start();
 * </pre>
 * <br>
 * Under normal operation, you would normally have two threads one that reads the underlying
 * transport and submits data to the GAL receiver and another that reads data from the GAL
 * receiver and submits data to the underlying transport.
 * <br>
 * A typical transport to GAL loop may look like the pseudocode below. Note that this example
 * assumes that you are reading from a buffered transport, it might be necessary to add your
 * own buffering layer if the underlying transport does not support arbitrary sized reads.
 * <pre>
 *      uint8_t header[FRAME_HEADER_MIN_LENGTH];
 *      while (!stop) {
 *          readFromTransport(header, sizeof(header));
 *          int len = galReceiver.getAdditionalBytesToRead(header);
 *          memcpy(buf, header, sizeof(header);
 *          readFromTransport(buf + sizeof(header), len);
 *          mGalReceiver.queueIncoming(buf, len);
 *      }
 * </pre>
 * <br>
 * A typical GAL to transport loop may look like this:
 * <pre>
 *      IoBuffer buf;
 *      while (!stop) {
 *          galReceiver.getEncodedFrame(&buf);
 *          writeToTransport(buf.raw(), buf.size());
 *      }
 * </pre>
 * <br>
 * The shutdown sequence would typically look like this. The most common reason to tear down
 * would be a disconnection event either caused by an unplug or by a callback signalling an
 * unrecoverable error.
 * <pre>
 *      galReceiver.prepareShutdown(); // This will unblock any blocked threads.
 *      // Stop transport threads.
 *      galReceiver.shutdown();
 * </pre>
 */
class GalReceiver {
public:
    GalReceiver() : mController(&mMessageRouter) { }
    /**
     * Call this method to initialize the receiver library. This must be called before any other
     * methods on the library.
     */
    bool init(const shared_ptr<IControllerCallbacks>& controllerCallbacks);
    /**
     * Call this when you want to send out the VERSION_REQUEST message to the phone. All services
     * must be registered before this.
     */
    void start();
    /**
     * Call this method to unblock any threads that may be waiting on getEncodedFrame().
     */
    void prepareShutdown();
    /**
     * Call this method to shutdown the receiver. It is okay to call init again once this method
     * has returned.
     */
    void shutdown();
    /**
     * This method should be called when new data is available for processing. The data is copied
     * so it may be freed once this method returns.
     * @param raw The raw data that was received.
     * @param len The size of data in bytes.
     * @return STATUS_SUCCESS if the data was processed successfully, an approriate error code
     *         otherwise.
     */
    int queueIncoming(void* raw, size_t len);
    /**
     * Acquire a buffer that can be passed on to queueIncoming.
     * @return A shared_ptr to an IoBuffer of the requested size.
     */
    shared_ptr<IoBuffer> allocateBuffer(size_t size);
    /**
     * This method should be called when there is new data available for processing. With this
     * method, you transfer ownership of the buffer to the receiver library so you should not
     * use it after calling this method.
     * @param buf The buffer containing the data.
     * @return STATUS_SUCCESS if the data was processed successfully, an approriate error code
     *         otherwise.
     */
    int queueIncoming(const shared_ptr<IoBuffer>& buf);
    /**
     * When data is available on the underlying transport, call this method with the first 4 bytes
     * to determine how many more bytes should be read to reach the end of the frame. The data
     * read along with these 4 bytes should be passed into queueIncoming subsequently.
     * @param buf A buffer with the data that belongs to the header.
     * @return How many more bytes should be read to get the complete message.
     */
    int getAdditionalBytesToRead(unsigned char buf[FRAME_HEADER_MIN_LENGTH]);
    /**
     * Call this method to obtain a frame for transmission. Note that this method blocks until a
     * frame is actually available. The IoBuffer that is passed in will be populated with the
     * frame data.
     * @param buf The io buffer that should be populated with the frame data.
     * @return true on success, false otherwise.
     */
    bool getEncodedFrame(IoBuffer* buf);
    /**
     * Call this function to register a new 'Service' with the receiver. This allows the receiver
     * to advertise it in the ServiceDiscoveryResponse and for the phone to open channels to it.
     * @param endpoint The service to register.
     * @return true on success, false otherwise.
     */
    bool registerService(ProtocolEndpointBase* endpoint);
    /**
     * Call this function to set up identity information of the car. This must be called prior to
     * calling start(). Send empty strings for any fields that you cannot provide.
     * @param make The make of the car.
     * @param model Car model.
     * @param year Production year.
     * @param id A unique identifier for this car. Ideally, there should be a way for the user to
     *           have this id regenerated.
     */
    void setIdentityInfo(const string& make, const string& model, const string& year,
            const string& id) {
        mController.setIdentityInfo(make, model, year, id);
    }
    /**
     * Set up information about the head unit. Send empty strings for any fields that you cannot
     * provide.
     * @param huMake The make of the head unit.
     * @param huModel The model of the head unit.
     * @param huSwBuild The software build of the head unit.
     * @param huSwVersion The software version of the head unit.
     */
    void setHeadUnitInfo(const string& huMake, const string& huModel,
            const string& huSwBuild, const string& huSwVersion) {
        mController.setHeadUnitInfo(huMake, huModel, huSwBuild, huSwVersion);
    }
    /**
     * Sets the driver position - left/right/center. This influences the layout of the screen.
     * This must be called before start().
     * @param position See the DriverPosition protocol buffer enum for details.
     */
    void setDriverPosition(int position) {
        mController.setDriverPosition(position);
    }
    /**
     * Sets the value sent to the MD to configure its behaviour during projection.
     * This must be called before start().
     * @param sessionConfiguration the bitmask to send to the MD
     */
    void setSessionConfiguration(int sessionConfiguration) {
        mController.setSessionConfiguration(sessionConfiguration);
    }
    /**
     * Set the credentials used in the ssl handshake. Must be called before calling start().
     * @param rootCert The root of trust to be used while validating a peer certificate.
     * @param clientCert The client certificate to be used in the handshake.
     * @param privKey The private key corresponding to the client certificate.
     * @return true on success, false otherwise.
     */
    bool setClientCreds(const string& rootCert, const string& clientCert, const string& privKey) {
        return mController.setClientCreds(rootCert, clientCert, privKey);
    }
    /**
     * Ping the other side to see if it is alive. This is mostly for diagnosing connectivity issues.
     * @param timestamp The local timestamp. This is repeated by the other side in the response
     *        so it can be used to compute round trip delays.
     * @param bugReport Request the other end to generate a bug report locally. The other end may
     *        choose to ignore this request.
     */
    void sendPingRequest(int64_t timestamp, bool bugReport) {
        mController.sendPingRequest(timestamp, bugReport);
    }
    MessageRouter* messageRouter() { return &mMessageRouter; }
    /**
     * Set the navigation focus. This is used to prevent the case where the user is offered turn
     * by turn directions from both the native navigation application on the head unit and by the
     * connected phone. When the phone requests navigation focus, this method should be used to
     * grant it, or if the user starts up the native navigation application, this method can be
     * used to acquire it. Navigation focus should only be changed as a direct consequence of a
     * user action.
     *
     * <p>Navigation focus must always be set in response to a navigation focus request
     * (IControllerCallbacks::navigationFocusCallback()) from the mobile device (even if there
     * is no change in navigation focus).
     *
     * @param type Can be NAV_FOCUS_NATIVE or NAV_FOCUS_PROJECTED.
     */
    void setNavigationFocus(int type) {
        mController.setNavigationFocus((NavFocusType) type);
    }

    /**
     * Send ByeByeRequest to phone.
     */
    void sendByeByeRequest(int code) {
        mController.sendByeByeRequest((ByeByeReason) code);
    }

    /**
     * Send ByeByeResponse as a response to ByeByeRequest.
     */
    void sendByeByeResponse() {
        mController.sendByeByeResponse();
    }

    /**
     * Set audio focus.
     */
    void setAudioFocus(AudioFocusStateType focusState, bool unsolicited) {
        mController.setAudioFocus(focusState, unsolicited);
    }
    /**
     * Use this api to set the time that will be used to verify the validity of the certificate
     * presented by the phone. You should only use this if your system does not run wall time.
     * This should be called after init() but before start() if you wish to override the time,
     * you should be able to call it any number of times after without any effect.
     * @param The number of seconds elapsed since Jan 1 1970 UTC 00:00:00.
     */
    void setCertificateVerificationTime(time_t time) {
        mController.setCertificateVerificationTime(time);
    }

private:
    MessageRouter mMessageRouter;
    ChannelManager mChannelManager;
    Controller mController;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_GAL_RECEIVER_H
